/* Program to compute trajectory of object projected from a ramp towards
   a ledge, and to tell if it lands correctly. */

#include <stdio.h>
#include <math.h>
#include "vector2.h"
#define FALSE (0)
#define TRUE (1)

float RAD_FACTOR;                  /* Degree to radian conversion factor */
#define FPS_FACTOR (5280.0/3600.0) /* mph to ft/sec conversion factor */

vector2 get_init_velocity(void);
vector2 get_ledge_edge(void);
void produce_results(vector2 p, vector2 u);

main() {
    vector2 p,                     /* Position of ledge edge from ramp */
            u;                     /* Initial velocity */
    RAD_FACTOR = atan(1.0)/45.0;   /* Initialize using the ratio of 45 */
                                   /* degrees to its radian value. */

    u = get_init_velocity();
    p = get_ledge_edge();
    produce_results(p, u);         /* Computes and prints solution */
    return 0;
}

/**** Functions to get initial settings: Compute from user input ****/

vector2 get_init_velocity(void) {
    vector2 u;
    float speed, angle;
    printf("Enter vehicle speed (mph): ");
    scanf("%f", &speed);
    printf("Enter angle of ramp to horizontal (degrees): ");
    scanf("%f", &angle);
    u = v2Polar(speed*FPS_FACTOR, angle*RAD_FACTOR);
    return u;
}

vector2 get_ledge_edge(void) {
    vector2 p;
    printf("Enter distance of ledge from ramp (feet): ");
    scanf("%f", &p.x);
    printf("Enter height of ledge above ramp (feet): ");
    scanf("%f", &p.y);
    return p;
}

/**** produce_results: Compute and display behaviour of cycle. ****/

void produce_results(vector2 p, vector2 u) {
    vector2 g = {0, -32.0};       /* gravity (constant) */
    #define DELTA_T (0.1)         /* the length of each time interval */
    float t = 0;                  /* time since cycle took off. */
    vector2 v;                    /* velocity at end of time interval */
    vector2 s;                    /* distance travelled in time interval */
    vector2 a;                    /* Assumed constant acceleration during
                                     a short time interval */
    vector2 total_s = {0.,0.};    /* distance from origin at time t. */
    int over_ledge = FALSE;       /* Not yet over ledge at the start */

    /* Print headings */
    printf(
        "\n Time   Horizontal    Vertical    Horizontal   Vertical\n"
        " (sec) Position(ft) Position(ft)  Speed(mph)  Speed(mph)\n"
    );

    /* while the cycle is in the air, either it has not reached the ledge,
       or it has reached it, but is above it. */
    while (total_s.x < p.x  ||  total_s.y > p.y) {
        /* Print details */
        printf(
            "%5.1f   %8.2f     %8.2f     %8.2f     %7.2f",
            t, total_s.x, total_s.y, u.x/FPS_FACTOR, u.y/FPS_FACTOR
        );
        if ( (! over_ledge) && total_s.x >= p.x) {
            printf("  Now over the ledge!");
            over_ledge = TRUE;
        }
        printf("\n");

        /* Recalculate a for the next short time interval */
        a = v2Sub(g, sv2Mul(0.02, u));   /* a = g - 0.02u */

        t += DELTA_T;                    /* Advance the total time */

        /* Compute distance travelled in time interval */
        s = v2Add(sv2Mul(DELTA_T, u), sv2Mul(0.5*DELTA_T*DELTA_T, a));
                                         /* s = ut + .5att */

        total_s = v2Add(total_s, s);     /* add s to total */

        /* Compute velocity at end of time interval */
        v = v2Add(u, sv2Mul(DELTA_T, a));/* v = u + at */
        u = v;                           /* This v is next iteration's u */
    }

    /* Compute and print final outcome information. */
    if (! over_ledge) {
        printf(
            "\nCycle FAILED to reach ledge! Missed by %3.1f ft.\n",
            p.y - total_s.y
        );
    } else {
        printf(
            "\nCycle made the ledge! Landed at angle %2.0f deg.\n",
            /* Convert landing angle to degrees, then make positive. */
            - v2Arg(v)/RAD_FACTOR
        );
    }
}
