/* HELPADT: Help database Abstract Data Type definition. */

#include <stdio.h>              /* for type fpos_t */
#define NAMESIZE (11)           /* Buffer size for help item name */
#define DESCRIPTSIZE (51)       /* Buffer size for a short description */

/*############### DATATYPES ###############*/

/*** help_id: Information needed to locate a full help entry. ***/
typedef struct {
    fpos_t where;
    int size;
} help_id;

/*** helpndx: Contains a help index entry ***/
typedef struct {
    char name[NAMESIZE];          /* the one-word name */
    char descript[DESCRIPTSIZE];  /* the short description */
    help_id loc;                  /* location info. for full help entry */
} helpndx;

typedef struct HELPSYS HELPSYS;   /* Note: an incomplete type */

/*############### FUNCTIONS ###############*/

/****** help_open(index, helpbase, retain) - opens a help database *******
 On entry: index is the name of a help index file, helpbase is the name of
           the corresponding full help database.  retain is the number of
           index entries the help system should secretly store in memory
           to speed access to the data.
 On exit:  If successful, returns a pointer to a HELPSYS structure
           identifying the opened help database.  On failure, returns NULL.
*/
HELPSYS *help_open(
    char /*index*/ [],
    char /*helpbase*/ [],
    int /*retain*/
);

/****** help_close(helpsys) - closes a help database ******
 On entry: helpsys is a HELPSYS pointer to an open help database.
 On entry: The database is closed and all storage released.
*/
void help_close(HELPSYS * /*helpsys*/);

/****** help_next(helpsys) - supply the next index entry ******
 On entry: helpsys points to an open help database.
 On exit:  On success, returns a pointer to a structure of type helpndx,
           containing an index entry for the 'next' help item; this pointer
           is only valid until the next 'help_next' or 'help_name' function
           call. Successive calls advance item-by-item through the database.
           A call to help_next after help_open will access the first item
           in the help database.  Returns NULL if no more items are found.
*/
helpndx *help_next(HELPSYS * /*helpsys*/);

/****** help_reset(helpsys) - reset the database ******
 On entry: helpsys refers to an open help database.
 On exit:  A following call to help_next will access the first help item.
*/
void help_reset(HELPSYS * /*helpsys*/);

/****** help_name(helpsys, itemname) - supply matching index entry ******
 On entry: helpsys points to an open help database.
 On exit:  On success, returns a pointer to a structure of type helpndx,
           containing an index entry for the help item with name itemname;
           this pointer is only valid until the next 'help_next' or
           'help_name' function call.  Returns NULL if item is not found.
*/
helpndx *help_name(HELPSYS * /*helpsys*/, char * /*itemname*/);

/****** help_message(helpsys, location) - Retrieve a full help entry ******
 On entry: helpsys points to an open help database.  location is the loc
           field from a helpndx obtained from help_next or help_name.
 On exit:  Returns a pointer to a string containing the full help message.
           N.B.: This pointer should be free'd when no longer required.
           If no entry is found, returns NULL.
*/
char *help_message(HELPSYS * /*helpsys*/, help_id /*location*/);
