/* HELPADT: Help database Abstract Data Type implementation. */

#include <helpadt.h>
#include <stdlib.h>
#include <string.h>

struct HELPSYS {
    FILE *indexf, *fullf;           /* For opened FILE pointers */
    helpndx ndxitem;                /* For one index file entry */
    fpos_t firstindex;              /* Index file position of first entry */
    fpos_t nextindex;               /* Index file position of next entry */
};

static int getstr(FILE *f, char buf[], int max);   /*
^^^^^^ See chapter 18 for meaning of static - ignore for now. */
static helpndx * help_at(HELPSYS *h, fpos_t *place);

HELPSYS *help_open(char index[], char helpbase[], int retain) {
    HELPSYS *hlp; char hdr[FILENAME_MAX];

    /* First try for space for the HELPSYS structure */
    if ((hlp = (HELPSYS*)malloc(sizeof(HELPSYS))) == NULL) {
        return NULL;            /* malloc failed. */
    }

    /* Open index file; if fail, free HELPSYS memory & return */
    if ((hlp->indexf = fopen(index, "rb")) == NULL) {
        free(hlp);              /* opening index file failed */
        return NULL;
    }

    /* Now error-check index file header & try to open full file;
       if any step fails, free space & close index file, & return.
    */
    if (
           getc(hlp->indexf) != '!'   /* First char isn't '!'? */
        || getc(hlp->indexf) != '!'   /* second char isn't '!'? */
        || getstr(hlp->indexf, hdr, FILENAME_MAX)   /* string missing? */
        || strcmp(helpbase, hdr) != 0 /* Is string in file the same name as
                                         the help database? */
        || (hlp->fullf = fopen(helpbase, "r")) == NULL
                                      /* Can we open the help database? */
    ) {
        fclose(hlp->indexf);     /* opening full help file failed */
        free(hlp);
        return NULL;
    }

    /* We have now read past the header of the index file; Record this
       file position for help_next and help_reset.
    */
    if (fgetpos(hlp->indexf, &hlp->nextindex) != 0) {
        fclose(hlp->fullf);     /* failed */
        fclose(hlp->indexf);
        free(hlp);
        return NULL;
    }
    hlp->firstindex = hlp->nextindex;

    return hlp;                 /* Success: return address of hlp */
}

/****** help_close(helpsys) - closes a help database ******
 On entry: helpsys is a HELPSYS pointer to an open help database.
 On entry: The database is closed and all storage released.
*/
void help_close(HELPSYS *helpsys) {
    fclose(helpsys->indexf);
    fclose(helpsys->fullf);
    free(helpsys);
}

helpndx *help_next(HELPSYS *h) {
    /* Uses help_at to do all the hard work */
    return help_at(h, &h->nextindex);
}
/*** help_at(h, place) - loads an index entry from the specified place ***
 On entry: h is a HELPSYS*, *place is an fpos_t in the index file.
 On exit:  *place updated to next entry in index file.
           Function return as for help_next.
*/
static helpndx * help_at(HELPSYS *h, fpos_t *place) {
    /* Set position, input name, short description, location and size,
       then test position & record.  By writing as conditions in an if
       statement, if any step fails, the rest are abandoned, and
       the error code is executed.
    */
    if (
        fsetpos(h->indexf, place) ||
        getstr(h->indexf, h->ndxitem.name, NAMESIZE) ||
        getstr(h->indexf, h->ndxitem.descript, DESCRIPTSIZE) ||
        fread(&h->ndxitem.loc.where, sizeof(fpos_t), 1, h->indexf) < 1 ||
        fread(&h->ndxitem.loc.size, sizeof(int), 1, h->indexf) < 1 ||
        fgetpos(h->indexf, place)
    ) {
        return NULL;
    } else {
        return &h->ndxitem;
    }
}

void help_reset(HELPSYS *helpsys) {
    helpsys->nextindex = helpsys->firstindex;
}

helpndx *help_name(HELPSYS *helpsys, char *itemname) {
    fpos_t indexplace; helpndx *indexentry;
    /* Load each entry in the index file until EOF or correct one found. */
    indexplace = helpsys->firstindex;
    do {
        indexentry = help_at(helpsys, &indexplace);
    } while (indexentry!=NULL && strcmp(indexentry->name, itemname)!=0);
    return indexentry;
}

char *help_message(HELPSYS *hlp, help_id location) {
    char *message; int ch, i;
    message = (char*)malloc((location.size + 1));
    if (message == NULL) {
        return NULL;                             /* Cannot malloc buffer */
    }
    /* Now set position, read data; if successful, append nul. */
    if (fsetpos(hlp->fullf, &location.where)) { /* Can't locate */
        return NULL;
    }
    for (i=0; i<location.size; i++) {           /* Input characters */
        message[i] = ch = getc(hlp->fullf);
    }
    message[location.size] = '\0';
    /* Now check on file consistency.  (Exercise: why does it work?) */
    if (ch == EOF || (ch=getc(hlp->fullf)) != ':' && ch != EOF) {
        fprintf(stderr, "Error: help system inconsistent; use helpmake\n");
        return NULL;
    }
    return message;
}

/****** getstr: input a nul-terminated string ******
 On entry: f is an open input file, buf is a buffer at least max bytes long.
 On exit:  A nul-terminated string is input from f; as much as will fit is
           stored in buf, always including a trailing nul.  Returns 0 for
           success, 1 for failure.
*/
static int getstr(FILE *f, char buf[], int max) {
    int i=0, ch, limit = max - 1;
    /* The following loop reads the entire string, but stops advancing
       through buf after the final place - if overfull, succeeding chars
       overwrite each other in the final element of buf. */
    while ((ch = getc(f)) > '\0') {     /* loop until nul or EOF */
        buf[i] = ch;
        if (i < limit) {                /* still room left? */
            i++;                        /* yes - next char in next place */
        }
    }
    if (i <= limit) {                   /* Input within string limits? */
        buf[i] = '\0';                  /* nul after last data byte */
    } else {                            /* String completely full */
        buf[limit] = '\0';              /* nul in final char */
    }
    return (ch == EOF);         /* Return 1 if ch== EOF, 0 otherwise */
}
