/* A name tag function suitable for inclusion in a function library.

The solution presented here allows the caller to print a tag with a
name, and a numeric value, along with an explanation for the numeric
value.  For example:

************************************************
************************************************
**                                            **
**                 John Smith                 **
**                                            **
**                 Delegate 47                **
**                                            **
************************************************
************************************************

Here, "John Smith" is the name, 47 the number, and "Delegate" the
explanation.  Here is the function heading and specification:  */

/************ print_tag: a function to print a name tag *************
On entry: Name is a string containing a name, Description is a string
          containing a description for a numeric quantity, and Value
          is that quantity.  Height is the height of the tag, Width the
          width, Hborder the number of rows of stars top and bottom,
          Vborder the number of columns of stars left and right, and
          Gap the number of blank rows to follow the tag (to ensure
          a correct fit on things such as sticky labels).
          Height shall be >= than 2*Hborder+2,
          Width shall be >= 2*Vborder + space for longest Name or
          (Description+Value+1).
          Value shall be in the range 0 to 9999.
On exit:  A tag as described by the parameters shall have been printed.
          The Name shall be centred on a line, and the Description+Value
          likewise.  Available blank lines shall be distributed evenly
          above, below, and in between, these two lines.
*/

#include <stdio.h>
#include <string.h>
/* Subsidiary function declarations:  (Note: the use of 'static' is
explained in Chapter 18.  The program is still correct if omitted.) */

static void repeat_char (char which_char, int number);
static void print_vacant (int no_of_lines, int width, int borderwidth);
static void print_name_line (char Name[], int width, int borderwidth);
static void print_value_line (
    char Description[], int Value, int width, int borderwidth
);

void print_tag (
    char Name[], char Description[], int Value,
    int Height, int Width, int Hborder, int Vborder, int Gap
) {
    int first_space;    /* space between top border and name line */
    int middle_space;   /* space between name line and value line */
    int last_space;     /* space between value line and bottom border */
    int i;              /* temporary calculations */

    /* Compute vertical spacing: */
    i = Height - 2 * Hborder - 2;   /* The available no. of blank lines */
    middle_space = i / 3;
    first_space = (i - middle_space) / 2;
    last_space = i - middle_space - first_space;

    /* Print top border: */
    for (i = 0; i < Hborder; i++) {
        repeat_char('*', Width); putchar('\n');
    }

    /* Print first vacant part of tag: */
    print_vacant(first_space, Width, Vborder);

    print_name_line(Name, Width, Vborder);

    /* Print middle vacant part of tag: */
    print_vacant(middle_space, Width, Vborder);

    print_value_line(Description, Value, Width, Vborder);

    /* Print bottom vacant part of tag: */
    print_vacant(last_space, Width, Vborder);

    /* Print bottom border: */
    for (i = 0; i < Hborder; i++) {
        repeat_char('*', Width); putchar('\n');
    }

    /* Print gap between tags: */
    repeat_char('\n', Gap);
}

static void repeat_char (char which_char, int number) {
    /* Repeats the specified character, number times. */
    int i;
    for (i=0; i<number; i++) {
        putchar(which_char);
    }
}

static void print_vacant (int no_of_lines, int width, int borderwidth) {
    /* Prints a 'vacant' piece of a tag (i.e. without horizontal border,
       name, or description).
    */
    int rows;
    for (rows=0; rows<no_of_lines; rows++) {
        /* Print one line: */
        repeat_char('*', borderwidth);
        repeat_char(' ', width - 2 * borderwidth);
        repeat_char('*', borderwidth);
        putchar('\n');
    }
}

static void print_name_line (char Name[], int width, int borderwidth) {
    /* Prints the tag line that contains the name. */
    int spacing;                /* the no. of blanks before the Name */

    /* Convert width to amount of blank space only: */
    width = width - 2 * borderwidth -strlen(Name);

    spacing = width / 2;        /* the spacing before the name */

    repeat_char('*', borderwidth);
    repeat_char(' ', spacing);
    printf("%s", Name);
    repeat_char(' ', width - spacing);
    repeat_char('*', borderwidth);
    putchar('\n');
}

static void print_value_line (
    char Description[], int Value, int width, int borderwidth
) {
    /* Prints the tag line that contains the Value. */
    int spacing;                /* the no. of blanks before the message */
    int value_width;            /* spacing needed for the value */

    /* Calculate spaces needed for the Value:  (There are better ways to do
       this, but they need features that haven't been studied yet.)
    */
    if (Value < 10) {
        value_width = 1;
    } else if (Value < 100) {
        value_width = 2;
    } else if (Value < 1000) {
        value_width = 3;
    } else {              /* (Recall Value should be < 10000.) */
        value_width = 4;
    }

    /* Convert width to amount of blank space only: */
    width = width - 2 * borderwidth - strlen(Description) - value_width - 1;

    spacing = width / 2;        /* the spacing before the message */

    repeat_char('*', borderwidth);
    repeat_char(' ', spacing);
    printf("%s %d", Description, Value);
    repeat_char(' ', width - spacing);
    repeat_char('*', borderwidth);
    putchar('\n');
}

/********** End of section for inclusion in function library **********/

/* Test main function; tries printing tags of various sizes. */

main () {
    print_tag("Marjory Banks", "Delegate", 398, 7, 50, 1, 5, 2);
    printf(
        "That was %s, %s %d, height %d, width %d, Hborder %d, Vborder %d, Gap %d\n\n",
        "Marjory Banks", "Delegate", 398, 7, 50, 1, 5, 2
    );
    print_tag("Carrington Smythe", "Seat", 9, 8, 20, 2, 1, 1);
    printf(
        "That was %s, %s %d, height %d, width %d, Hborder %d, Vborder %d, Gap %d\n\n",
        "Carrington Smythe", "Seat", 9, 8, 20, 2, 1, 1
    );
    print_tag("Carrington Smythe", "Seat", 9, 4, 15, 1, 1, 1);
    printf(
        "That was %s, %s %d, height %d, width %d, Hborder %d, Vborder %d, Gap %d\n\n",
        "Carrington Smythe", "Seat", 9, 4, 15, 1, 1, 1
    );
}
